<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Categories - ReviewPR</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Inter', Arial, sans-serif; background: #f8f9fa; }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 2rem; border-radius: 12px; margin-bottom: 2rem; }
        .card { background: white; border-radius: 12px; padding: 2rem; margin-bottom: 2rem; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
        .form-group { margin-bottom: 1.5rem; }
        .form-label { display: block; font-weight: 600; margin-bottom: 0.5rem; color: #374151; }
        .form-control { width: 100%; padding: 0.75rem; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1rem; }
        .form-control:focus { outline: none; border-color: #667eea; }
        .btn { padding: 0.75rem 1.5rem; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; text-decoration: none; display: inline-block; }
        .btn-primary { background: #667eea; color: white; }
        .btn-success { background: #10b981; color: white; }
        .btn-danger { background: #ef4444; color: white; }
        .btn-secondary { background: #6b7280; color: white; }
        .btn:hover { opacity: 0.9; transform: translateY(-1px); }
        .grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1.5rem; }
        .category-card { background: white; border-radius: 12px; padding: 1.5rem; box-shadow: 0 2px 4px rgba(0,0,0,0.1); border: 1px solid #e5e7eb; }
        .alert { padding: 1rem; border-radius: 8px; margin-bottom: 1rem; }
        .alert-success { background: #d1fae5; color: #065f46; border: 1px solid #a7f3d0; }
        .alert-error { background: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }
        .nav-links { margin-bottom: 2rem; }
        .nav-links a { margin-right: 1rem; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🏷️ Category Management System</h1>
            <p>Add, edit, and manage product categories</p>
        </div>

        <div class="nav-links">
            <a href="admin_dashboard.php" class="btn btn-secondary">← Back to Dashboard</a>
        </div>

        <?php
        // Handle form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                require_once 'config.php';
                $conn = getSecureConnection();

                if (isset($_POST['action'])) {
                    switch ($_POST['action']) {
                        case 'add':
                            $stmt = $conn->prepare("INSERT INTO categories (name, description, position, is_active) VALUES (?, ?, ?, 1)");
                            $stmt->bind_param("ssi",
                                $_POST['name'],
                                $_POST['description'],
                                $_POST['position']
                            );

                            if ($stmt->execute()) {
                                echo '<div class="alert alert-success">✅ Category added successfully!</div>';
                            } else {
                                echo '<div class="alert alert-error">❌ Error adding category: ' . $stmt->error . '</div>';
                            }
                            $stmt->close();
                            break;

                        case 'update':
                            $stmt = $conn->prepare("UPDATE categories SET name=?, description=?, position=?, is_active=? WHERE id=?");
                            $stmt->bind_param("ssiii",
                                $_POST['name'],
                                $_POST['description'],
                                $_POST['position'],
                                $_POST['is_active'],
                                $_POST['id']
                            );

                            if ($stmt->execute()) {
                                echo '<div class="alert alert-success">✅ Category updated successfully!</div>';
                            } else {
                                echo '<div class="alert alert-error">❌ Error updating category: ' . $stmt->error . '</div>';
                            }
                            $stmt->close();
                            break;

                        case 'delete':
                            $stmt = $conn->prepare("DELETE FROM categories WHERE id=?");
                            $stmt->bind_param("i", $_POST['id']);

                            if ($stmt->execute()) {
                                echo '<div class="alert alert-success">✅ Category deleted successfully!</div>';
                            } else {
                                echo '<div class="alert alert-error">❌ Error deleting category: ' . $stmt->error . '</div>';
                            }
                            $stmt->close();
                            break;
                    }
                }

                $conn->close();
            } catch (Exception $e) {
                echo '<div class="alert alert-error">❌ Database Error: ' . $e->getMessage() . '</div>';
            }
        }
        ?>

        <!-- Add New Category Form -->
        <div class="card">
            <h2>➕ Add New Category</h2>
            <form method="POST">
                <input type="hidden" name="action" value="add">

                <div class="grid">
                    <div class="form-group">
                        <label class="form-label">Category Name *</label>
                        <input type="text" name="name" class="form-control" required placeholder="e.g., Trustpilot Reviews">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Position (Order) *</label>
                        <input type="number" name="position" class="form-control" required value="1" min="1">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea name="description" class="form-control" placeholder="Brief description of the category..."></textarea>
                </div>

                <button type="submit" class="btn btn-success">➕ Add Category</button>
            </form>
        </div>

        <!-- Existing Categories -->
        <div class="card">
            <h2>📋 Existing Categories</h2>

            <?php
            try {
                require_once 'config.php';
                $conn = getSecureConnection();

                // Ensure categories table exists
                $conn->query("CREATE TABLE IF NOT EXISTS categories (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    name VARCHAR(255) NOT NULL,
                    description TEXT,
                    position INT DEFAULT 1,
                    is_active TINYINT DEFAULT 1,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )");

                // Insert sample categories if table is empty
                $check = $conn->query("SELECT COUNT(*) as count FROM categories");
                if ($check && $check->fetch_assoc()['count'] == 0) {
                    $sample_categories = [
                        ['Trustpilot', 'Trustpilot review services', 1],
                        ['Google', 'Google review services', 2],
                        ['G2', 'G2 review services', 3],
                        ['Capterra', 'Capterra review services', 4],
                        ['Yelp', 'Yelp review services', 5],
                        ['Facebook', 'Facebook review services', 6],
                        ['Other', 'Other review services', 7]
                    ];

                    foreach ($sample_categories as $cat) {
                        $conn->query("INSERT INTO categories (name, description, position) VALUES ('$cat[0]', '$cat[1]', $cat[2])");
                    }
                }

                $result = $conn->query("SELECT * FROM categories ORDER BY position ASC, id ASC");

                if ($result && $result->num_rows > 0) {
                    echo '<div class="grid">';

                    while($category = $result->fetch_assoc()) {
                        echo '<div class="category-card">';
                        echo '<h3>' . htmlspecialchars($category['name']) . '</h3>';
                        echo '<p><strong>Position:</strong> ' . $category['position'] . '</p>';
                        echo '<p><strong>Status:</strong> ' . ($category['is_active'] ? 'Active' : 'Inactive') . '</p>';
                        if (!empty($category['description'])) {
                            echo '<p>' . htmlspecialchars(substr($category['description'], 0, 100)) . '...</p>';
                        }

                        echo '<div style="margin-top: 1rem;">';
                        echo '<button onclick="editCategory(' . $category['id'] . ')" class="btn btn-primary">✏️ Edit</button> ';
                        echo '<form method="POST" style="display: inline;" onsubmit="return confirm(\'Are you sure you want to delete this category?\')">';
                        echo '<input type="hidden" name="action" value="delete">';
                        echo '<input type="hidden" name="id" value="' . $category['id'] . '">';
                        echo '<button type="submit" class="btn btn-danger">🗑️ Delete</button>';
                        echo '</form>';
                        echo '</div>';

                        echo '</div>';
                    }

                    echo '</div>';
                } else {
                    echo '<p>No categories found. Add your first category above!</p>';
                }

                $conn->close();
            } catch (Exception $e) {
                echo '<div class="alert alert-error">❌ Error loading categories: ' . $e->getMessage() . '</div>';
            }
            ?>
        </div>
    </div>

    <!-- Edit Category Modal -->
    <div id="editModal" class="modal" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.4);">
        <div class="modal-content" style="background-color: #fefefe; margin: 5% auto; padding: 20px; border: 1px solid #888; width: 80%; max-width: 600px; border-radius: 12px;">
            <span class="close" onclick="closeModal()" style="color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer;">&times;</span>
            <h2>✏️ Edit Category</h2>
            <form method="POST" id="editForm">
                <input type="hidden" name="action" value="update">
                <input type="hidden" name="id" id="edit-id">

                <div class="form-group">
                    <label class="form-label">Category Name *</label>
                    <input type="text" name="name" id="edit-name" class="form-control" required>
                </div>

                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea name="description" id="edit-description" class="form-control"></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label">Position (Order) *</label>
                    <input type="number" name="position" id="edit-position" class="form-control" required min="1">
                </div>

                <div class="form-group">
                    <label class="form-label">Status</label>
                    <select name="is_active" id="edit-is-active" class="form-control">
                        <option value="1">Active</option>
                        <option value="0">Inactive</option>
                    </select>
                </div>

                <button type="submit" class="btn btn-primary">💾 Update Category</button>
                <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            </form>
        </div>
    </div>

    <script>
        function editCategory(id) {
            // Get category data via AJAX
            fetch('get_category.php?id=' + id)
                .then(response => response.json())
                .then(category => {
                    if (category) {
                        // Populate modal with category data
                        document.getElementById('edit-id').value = category.id;
                        document.getElementById('edit-name').value = category.name;
                        document.getElementById('edit-description').value = category.description || '';
                        document.getElementById('edit-position').value = category.position;
                        document.getElementById('edit-is-active').value = category.is_active;

                        // Show modal
                        document.getElementById('editModal').style.display = 'block';
                    } else {
                        alert('Category not found!');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading category data!');
                });
        }

        function closeModal() {
            document.getElementById('editModal').style.display = 'none';
        }

        // Close modal when clicking outside of it
        window.onclick = function(event) {
            const modal = document.getElementById('editModal');
            if (event.target == modal) {
                modal.style.display = 'none';
            }
        }

        // Handle form submission for editing
        document.getElementById('editForm').addEventListener('submit', function(e) {
            e.preventDefault(); // Prevent default form submission

            const formData = new FormData(this);

            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('Category updated successfully')) {
                    alert('Category updated successfully!');
                    closeModal();
                    location.reload(); // Refresh page to show updated data
                } else {
                    alert('Error updating category: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error updating category!');
            });
        });
    </script>
</body>
</html>
